<?php

// --------------------------------------------------------------------------------------------------
//   
//    SMS Gateway (BULK) integration with SMSLink.ro
//
//     - Version 1.9
//     - Requirements: PHP >= 5.0, CURL
//     - Optional: BZIP2, ZLIB for transmission compression
//     
//    CLIENT IMPLEMENTATION STARTS ON LINE: 410
//
//    Please note that SMS Gateway (BULK) can send only SMS up to 160 characters. For SMS longer 
//     than 160 characters, please use SMS Gateway (HTTP).
//
//    Important Change Log:
//
//        06-05-2016: 
//        - If PHP >= 5.5 (line 339)
//            1. Added support for CURLFile to replace the @ file upload modifier, as of PHP >= 5.5.
//            2. CURLOPT_SAFE_UPLOAD (available as of PHP >= 5.5) is set to default true to disable
//             the @ file upload modifier       
//           
//        28-01-2019:
//        - Enhancements for connection handling
//
//        06-03-2019:
//        - Changed constructor function from class name to  __construct()  due to PHP 7 deprecation
//          warning: "Methods with the same name as their class will not be constructors in a future
//          version of PHP."
//
// --------------------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------------------
//    
//        Class Implementation
//        
// --------------------------------------------------------------------------------------------------
class BulkSMSPackage
{    
    // ----------------------------------------------------------------------------------------------    
    //   Change the variabiles to match your account details    
    //        - Connection ID and Password can be generated at www.smslink.ro/sms/gateway/setup.php
    //          after authenticated with your Username or E-mail and Password
    // ----------------------------------------------------------------------------------------------        
    private $ConnectionID = "";            // SMS Gateway Connection ID
    private $Password = "";                // SMS Gateway Password
    
    private $Test = 0;                     // Set to 0 for Production Environment
                                           // Set to 1 for Test Environment (no SMS is sent)
                                           
    private $UseHTTPS = 0;                 // Set to 0 for HTTP connection to SMS Gateway
                                           // Set to 1 for HTTPS connection to SMS Gateway
    
    private $TemporaryDirectory = "/tmp";           
    
    // ----------------------------------------------------------------------------------------------    
    //   Recommendation: Do not change below this line WITHIN the CLASS
    // ----------------------------------------------------------------------------------------------
    public $Address      = "http://www.smslink.ro/sms/gateway/communicate/bulk.php";
    public $AddressHTTPS = "https://secure.smslink.ro/sms/gateway/communicate/bulk.php";
    
    public $RemoteMessageIDs = array();
    public $Error = "";
    public $Time = 0;
    
    private $Contents = array();    
    private $Status = 0;
        
    private $File = array(
            "Plain"      => "",
            "Compressed" => ""
        );
        
    private $Validation = array(
            "MD5" => array(
                "Plain"      => "",
                "Compressed" => ""
            )
        );

    protected $Version = 1.9;
    
    protected $Compressions = array(
            0 => array("CompressionID" => 0, "Compression" => "No Compression - Plain Text"),
            1 => array("CompressionID" => 1, "Compression" => "Compression ZLIB GZIP"),
            2 => array("CompressionID" => 2, "Compression" => "Compression BZIP2"),
            3 => array("CompressionID" => 3, "Compression" => "Compression LZF"),
        );
        
    private $Compression = 1;
    
    // -------------------------------------------------------------------------------------------------
    //   Constructor
    // -------------------------------------------------------------------------------------------------
    public function __construct($ConnectionID = "", $Password = "", $Test = -1)
    {
        if (strlen($ConnectionID) > 0)
            $this->ConnectionID = $ConnectionID;
            
        if (strlen($Password) > 0)
            $this->Password = $Password;
            
        if (($Test == 0) or ($Test == 1))
            $this->Test = $Test;
                
    }
    
    // -------------------------------------------------------------------------------------------------
    //   Destructor
    // -------------------------------------------------------------------------------------------------
    public function __destruct()
    {
    	$this->ConnectionID = "";
    	$this->Password     = "";
    	$this->Test         = 0;
    	
    }
    
    // -------------------------------------------------------------------------------------------------
    //   public function Compression
    //       - Description: Sets compression mechanism for the package.
    //       - Variabiles:
    //            - (int) $Compression accepts the following integer values:
    //                0 for Plain TEXT compression (no compression)
    //                1 for ZLIB GZIP compression (note that you may need ZLIB to be installed)
    //                2 for BZIP2 compresssion (note that you may need BZIP2 to be installed)
    //                3 for LZF compression
    //       - Returns: true on success or false on failure.
    // -------------------------------------------------------------------------------------------------
    public function Compression($Compression)
    {
        if ($this->Status == 0)
        {
            switch ($Compression)    
            {
                case 0:
                    $this->Compression = 0;
                    break;
                case 1:
                    $this->Compression = 1;
                    break;
                case 2:
                    $this->Compression = 2;
                    break;
                case 3:
                    $this->Compression = 3;
                    break;
                default:    
                    $this->Compression = 1;
                    break;
            }
            
            return true;
                
        }
        
        return false;
        
    }
    
    public function Compress($Content)
    {
        $Compressed = "";

        switch ($this->Compression) 
        {
            case 0:
                $Compressed = $Content;
                break;
            case 1:
                $Compressed = gzcompress($Content, 9);
                break;
            case 2:
                $Compressed = bzcompress($Content, 9);
                break;
            case 3:
                $Compressed = lzf_compress($Content);
                break;
        }
        
        return $Compressed;
        
    }
    
    // ---------------------------------------------------------------------------------------------------
    //   public function InsertMessage
    //       - Description: Inserts a SMS to your local queue before sending to SMSLink.ro.
    //       - Variabiles:
    //            - (int)    $LocalMessageID - Local Message ID, can be used for pairing delivery reports
    //            - (string) $Receiver       - Receiver mobile number, national format: 07XYZZZZZZ
    //            - (string) $Sender         - Sender alphanumeric string for SMS:
    //
    //              numeric    - sending will be done with a shortcode (ex. 18xy, 17xy)
    //              SMSLink.ro - sending will be done with SMSLink.ro (use this for tests only)
    //
    //              Any other preapproved alphanumeric sender assigned to your account:
    //                - Your alphanumeric sender list:        http://www.smslink.ro/sms/sender-list.php
    //                - Your alphanumeric sender application: http://www.smslink.ro/sms/sender-id.php
    //
    //              Please Note:
    //                - SMSLink.ro  sender should be used only for testing and is not  recommended to be 
    //                  used in production. Instead, you should  use numeric sender or your alphanumeric   
    //                  sender, if you have an alphanumeric sender activated with us.                        
    //                - If you set an alphanumeric sender for a mobile number that is in a network where  
    //                  the alphanumeric  sender has not been activated,  the system will override  that  
    //                  setting with numeric sender.
    //
    //            - (string) $Message             - Text SMS, up to 160 alphanumeric characters.
	//
    //                Recommended to be used with GSM7 IA5 alphabet (QWERTY characters).
    //                Invalid characters that will be stripped: \r\n \r \n \t ; and '
    //
    //            - (int)    $TimestampProgrammed - Should be 0 (zero) for immediate sending or other  
    //                                              UNIX timestamp in the future for future sending
    //
    //       - Returns: true on success or false on failure.
    // ---------------------------------------------------------------------------------------------------
    public function InsertMessage($LocalMessageID, $Receiver, $Sender, $Message, $TimestampProgrammed = 0)
    {
        if (!is_numeric($LocalMessageID))
            return false;
            
        if (!is_numeric($Receiver))
            return false;
            
        if (strlen($Receiver) != 10)
            return false;
            
        $Message = str_replace(array("\r\n", "\r", "\n", "\t", ";", "'"), "", $Message);
        $Message = substr(trim($Message), 0, 160);
        
        $this->Contents[] = array(
                "LocalMessageID"       => $LocalMessageID,
                "Receiver"               => $Receiver,
                "Sender"               => $Sender,
                "Message"               => $Message,
                "TimestampProgrammed" => $TimestampProgrammed
            );
            
        return true;
            
    }
    
    // ------------------------------------------------------------------------------------------------------
    //   public function RemoveMessage
    //       - Description: Removes a SMS from your local queue before sending to SMSLink.ro.
    //       - Variabiles:
    //            - (int)    $LocalMessageID      - Local Message ID
    //       - Returns: void
    // ------------------------------------------------------------------------------------------------------
    public function RemoveMessage($LocalMessageID)
    {
        foreach ($this->Contents as $MessageKey => $MessageContent)
            if ($MessageContent["LocalMessageID"] == $LocalMessageID)
                unset($this->Contents[$MessageKey]);
                
    }
    
    // ------------------------------------------------------------------------------------------------------
    //   public function PackageSize
    //       - Description: Counts the size of your local package and returns the total.
    //       - Returns: (int)
    // ------------------------------------------------------------------------------------------------------
    public function PackageSize()
    {
        return sizeof($this->Contents);
        
    }
    
    public function MicrotimeFloat()
    {
        list($usec, $sec) = explode(" ", microtime());
        return ((float)$usec + (float)$sec);
        
    }
    
    // ------------------------------------------------------------------------------------------------------
    //   public function SendPackage
    //     - Description: Sends the SMS package to SMSLink.ro for immediate processing and sending.
    //     - Returns: true on succes or false on failure
    //                If returns false, Error is written in $BulkSMSPackage->Error
    // ------------------------------------------------------------------------------------------------------
    public function SendPackage()
    {
        $TimeStart = $this->MicrotimeFloat();
        
        $this->RemoteMessageIDs = array();
        $this->Error = "";
        
        if (($this->Status == 0) and (sizeof($this->Contents) > 0))
        {
            // --------------------------------------------------------------------------------------
            //   Create TEXT file
            // --------------------------------------------------------------------------------------
            $Temp = array();
            foreach ($this->Contents as $MessageKey => $MessageContent)
                $Temp[] = implode(";", $MessageContent);

            $this->File["Plain"] = implode("\r\n", $Temp);
            $this->Validation["MD5"]["Plain"] = md5($this->File["Plain"]);
            
            // --------------------------------------------------------------------------------------
            //   Create COMPRESSED file
            // --------------------------------------------------------------------------------------
            $this->File["Compressed"] = $this->Compress($this->File["Plain"]);
            $this->Validation["MD5"]["Compressed"] = md5($this->File["Compressed"]);
            
            // --------------------------------------------------------------------------------------
            //   Transfer to SMSLink.ro
            // --------------------------------------------------------------------------------------
            $TemporaryFile = tempnam($this->TemporaryDirectory, "sms-package-");
            
            if ($TemporaryFile != false)
            {
                file_put_contents($TemporaryFile, $this->File["Compressed"]);
                
                $Transfer = array(
                        "connection_id"  => $this->ConnectionID,
                        "password"       => $this->Password,
                        "test"           => $this->Test,
                        "Compression"    => $this->Compression,
                        "MD5Plain"       => $this->Validation["MD5"]["Plain"],
                        "MD5Compressed"  => $this->Validation["MD5"]["Compressed"],
                        "SizePlain"      => strlen($this->File["Plain"]),
                        "SizeCompressed" => strlen($this->File["Compressed"]),
                        "Timestamp"      => date("U"),
                        "Buffering"         => 1,
                        "Version"        => $this->Version,                        
                        "Receivers"      => sizeof($this->Contents),
                        "Package"        => "@".$TemporaryFile
                    );
                    
                $EndpointURL = $this->Address;
                if ($this->UseHTTPS == 1)
                    $EndpointURL = $this->AddressHTTPS;
                
                $ch = curl_init($EndpointURL."?timestamp=".date("U"));

                curl_setopt($ch, CURLOPT_POST, 1);

                // ----------------------------------------------------------------------------------
                //  As of PHP 5.5, CURLOPT_SAFE_UPLOAD set to false will issue a deprecation message 
                //  As of PHP 5.5, @ upload modifier is replaced with new CURLFile()
                // ----------------------------------------------------------------------------------
                if ((version_compare(PHP_VERSION, '5.5') >= 0)) 
                {
                    $Transfer["Package"] = new CURLFile($TemporaryFile);
                    curl_setopt($ch, CURLOPT_SAFE_UPLOAD, true);
                }

                curl_setopt($ch, CURLOPT_POSTFIELDS, $Transfer);
                curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
                curl_setopt($ch, CURLOPT_HEADER, 0);
                curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
                
                curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 0);
                curl_setopt($ch, CURLOPT_TIMEOUT, 0);
                
                // ----------------------------------------------------------------------------------
                //  For increased performance, we recommend setting CURLOPT_SSL_VERIFYPEER and 
                //  CURLOPT_SSL_VERIFYHOST to false, as in the implementation below.                
                // ----------------------------------------------------------------------------------
                if (strpos($EndpointURL, "https://") !== false)
                {
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, false);
                }

                $Response = explode(";", curl_exec($ch));
                
                if ($Response[0] == "MESSAGE")
                {            
                    $Temp = explode(",", $Response[3]);
                    for($i = 0; $i < sizeof($Temp); $i++)
                    {
                        $Variabiles = explode(":", $Temp[$i]);
                        $this->RemoteMessageIDs[$Variabiles[0]] = array(
                                "LocalMessageID"  => $Variabiles[0],
                                "RemoteMessageID" => $Variabiles[1],
                                "Status"          => $Variabiles[2]
                            );
                    }
                    
                    $this->Status = 1;
                    $TimeEnd = $this->MicrotimeFloat();
                    $this->Time = round($TimeEnd - $TimeStart, 2);
                    
                    return true;
                    
                }
                else 
                {
                    $this->Error = implode(";", $Response);
                        
                }
                
                curl_close($ch);

            }
            
        }
        
        return false;
        
    }
    
}

// ------------------------------------------------------------------------------------------------------
//    
//        Client Implementation
//
// ------------------------------------------------------------------------------------------------------
//   Class Initialize / Create SMS Package
// ------------------------------------------------------------------------------------------------------
$BulkSMSPackage = new BulkSMSPackage();

// ------------------------------------------------------------------------------------------------------
//   Insert SMS into SMS Package
//        bool $BulkSMSPackage->InsertMessage(MessageID, Receiver, Sender, Message);
//         - Returns: true on succes or false on failure
//    
//     Please Note:
//       - SMSLink.ro sender should be used only for testing and is not recommended to be used in 
//         production. Instead, you should use numeric sender or your alphanumeric sender, if you 
//         have an alphanumeric sender activated with us.
//       - If you set an alphanumeric sender for a mobile number that is in a network where the 
//         alphanumeric sender has not been activated, the system will override that setting with 
//         numeric sender.
// ------------------------------------------------------------------------------------------------------
$BulkSMSPackage->InsertMessage(1, "0723178275", "numeric", "Test SMS 1");
$BulkSMSPackage->InsertMessage(2, "0723178275", "numeric", "Test SMS 2");
$BulkSMSPackage->InsertMessage(3, "0723178275", "numeric", "Test SMS 3");

// ------------------------------------------------------------------------------------------------------
//   Send Package to SMSLink.ro
//      bool $BulkSMSPackage->SendPackage();
//       - Returns: true on succes or false on failure
//       - If returns false, Error is written in $BulkSMSPackage->Error
// ------------------------------------------------------------------------------------------------------
$BulkSMSPackage->SendPackage();

// ------------------------------------------------------------------------------------------------------
//   Process SMSLink.ro response for each SMS (optional, you may also check the status in your account)
//        
//   Please note: 
//
//         $RemoteMessageIDs is a multidimensional array populated with the local message ID, remote
//         message id and message status, in the following descriptive format:
//            
//         $RemoteMessageIDs = array(
//               1 => array("LocalMessageID" => 1, "RemoteMessageID" => 1000000, "Status" => 3),
//               2 => array("LocalMessageID" => 2, "RemoteMessageID" => 1000001, "Status" => 3),
//               3 => array("LocalMessageID" => 3, "RemoteMessageID" => 1000002, "Status" => 3),
//               ...
//               N => array("LocalMessageID" => N, "RemoteMessageID" => 100000X, "Status" => 3),
//              );
//                
//         Use LocalMesageID[1..N] and RemoteMesageID[1..N] to link local and remote SMS and save the
//         message status.
//
//           The array content will be in the following format:
//            (numeric) LocalMesageID[1..N]   
//               Usually the message IDs from your side. Does not matter if they are unique or not in
//               our system.
//            (numeric) RemoteMesageID[1..N] 
//               The message IDs from our side. In production environment the values will be unique for 
//               each SMS and must be expected to be over hundred milions. In test environment the
//               values will always be 0 (zero) or a random number.
//            (numeric) MessageStatus
//               1 - Sender Failed
//               2 - Incorrect Number
//               3 - Success
//               4 - Internal Error
//
//   Below is an example of how you may process the $RemoteMessageIDs array after is populated with 
//   values.
//
// ------------------------------------------------------------------------------------------------------
$Counters = array();

if (sizeof($BulkSMSPackage->RemoteMessageIDs) > 0)
{
    foreach ($BulkSMSPackage->RemoteMessageIDs as $key => $value)
    {
        // --------------------------------------------------------
        //     Message Statuses
        //     - 1 - Sender Failed
        //     - 2 - Incorrect Number
        //     - 3 - Success
        //     - 4 - Internal Error
        // --------------------------------------------------------
        switch ($value["Status"])
        {
            // -------------------------------------------------
            //   1 - Sender Failed
            // -------------------------------------------------            
            case 1:
                $timestamp_send = -1;
                /* 
                
                    .. do something .. 
                    for example check the sender because is incorrect
                    
                */
                
                echo "Eroare transmitere pentru SMS cu ID: ".$value["LocalMessageID"]."<br />";
                
                $Counters["FailedSender"] = $Counters["FailedSender"] + 1;
                break;
            // -------------------------------------------------
            //   2 - Number Failed
            // -------------------------------------------------            
            case 2:
                $timestamp_send = -2;
                /* 
                
                    .. do something .. 
                    for example check the number because is incorrect    
                    
                */
                
                echo "Eroare transmitere pentru SMS cu ID: ".$value["LocalMessageID"]."<br />";
                
                $Counters["FailedNumber"] = $Counters["FailedNumber"] + 1;
                break;
            // -------------------------------------------------
            //   3 - Success
            // -------------------------------------------------
            case 3:
                $timestamp_send = date("U");
                /* 
                
                    .. do something .. 

                    Save in database the Remote Message ID, sent in variabile: $value["RemoteMessageID"].
                    Delivery  reports will  identify  your SMS  using our Message ID. Data type  for the 
                    variabile should be considered to be hundred milions (example: 220000000)                    
                    
                */
                
                echo "Transmitere cu succes pentru SMS cu ID: ".
                     $value["LocalMessageID"].
                     ", ID SMSLink: ".
                     $value["RemoteMessageID"].
                     "<br />";
                
                $Counters["Success"] = $Counters["Success"] + 1;
                break;
            // -------------------------------------------------
            //   4 - Internal Error
            // -------------------------------------------------
            case 4:
                $timestamp_send = -4;
                /* 
                
                    .. do something .. 
                    for example try again later

                    Internal Error may occur in the following circumstances:

                    (1) Number is Blacklisted (please check the Blacklist associated to your account), or
                    (2) An error occured at SMSLink (our technical support team is automatically notified)
                    
                */
                
                echo "Eroare transmitere pentru SMS cu ID: ".$value["LocalMessageID"]."<br />";
                
                $Counters["FailedGeneral"] = $Counters["FailedGeneral"] + 1;
                break;
        }
        
        $Counters["Total"] = $Counters["Total"] + 1;
        
    }
        
}
else
{
    echo "Eroare transmitere pachet SMS catre SMSLink: ".$BulkSMSPackage->Error."<br />";
    
}

?> 